#lang planet neil/sicp
;Exercise 1.8: Newton’s method for cube roots is based on the fact that if yy is an approximation to the cube root of xx, then a better approximation is given by the value
;(x/y2+2y)/3
;Use this formula to implement a cube-root procedure analogous to the square-root procedure.(In 1.3.4 we will see how to implement Newton’s method in general as an abstraction of these square-root and cube-root procedures.)
(define (cube x)
  (* x x x))

(define (improve y x)
  (/ (+ (/ x (* y y))
        (* 2 y)) 
     3))

(define (abs x)
  (if (> x 0)
      x
      [- 0 x]))

(define (goodEnough? guess x)
  (< (abs (- (cube guess)
             x))
     0.001))

(define (cuberoot-itr guess x)
  (if (goodEnough? guess x)
      guess
      [cuberoot-itr (improve guess x) x]))

(define (cubeRoot guess)
  (cuberoot-itr 13 guess))

----------test----------

(cube 10)
(improve 10 1000)
(abs -110)
(goodEnough? 1011 1000)
(cuberoot-itr 101 1000)
(cubeRoot 1000)

----------output----------

;1000
;10
;110
;#f
;10 2545236670499754663646699239176569567758438994224607159028123730672463404091405694835256796190098836920440394620457157655985158008428597866976765497151026629536420964552445164690979646768361833282831812658243117442684821402264364705495670205472698883100802580910077010157409104338427737930163197555566067922726112362795110495703827207285316463644258033840659164342673227422267193046849096428398138209171425329114424878370147304802755015408207741746450222850525642794179096233284359887793179914233159225146085922271366005182996623051792481767617727892049632329207237690800944454427811955517962135973436432668186821762065399261197420841987615350173409325708176060388460806798074984931537311811210966314271810983393482579379504672462246221243220960481340990236232099372751537521490726452248024614053707772652593592676230827863627183292012444267703593185591124926239608273703745411976692301533816739611260396002340988968795183966834661831328022982670821340307031116507355497870564587942712736644588679799778381246365364704439036308420732852062385313350980927396213819259294161207851119048134415649621290947824455838759018036241219584966149247780234988485545853897939124774895063976016093162179786721838379504098328136004647190253959565865148309339815059110782584737193917111510608768723429222942958022573115388649955427604903677381004326756999103255987575549779303053537185337663855952146848937292902219690308333676730521115593221135809533475247647249926334458353938739579875574724466202190298564359898442467974075114376364537331051639030425773599860510816944151158895062266408186012984696126264661828272530895009164897798542112972725781492901006918377289624181649671237874441322885450275619776695144412124451690560717375301306724689338998824804830731098581786263656781176294860273828473197546469125841596659638928256631011482337285636641381126726470638798885136806732135870430757215470498413650382028622572477727564089199759459542252824717742691547109432421162171108743888804618450782303828029881891409265297312760575161910369097836884625208527844586546445225795485341913567314839262931829942767059003844590711988635403577470274112104421996640934554132499012108437943558899730185842059329868337472195729183920985962188670673781613480977820332685854317271406647848218775502259964913733228356866372470961628753170737311533918860673675297697256297913379920136465715188191472514640173035013411585948779418227205213965904015747322006926448731519514959336460165425382142702705998587393251364789351402670092380892492014517196946447915847489828247893761241800382143157958792722239419631571232779866638675726864661639545506230134816244705439846833076195323317335283688304132076737479208160715125590737242724355526352786882012243366669586124938616435793054615054913490440067831983157504877289513994648518644706670742929035481551783599436960017020441520597347476823301142559043414964211949641712237976352879709956022046609963907256684284268791650764209155485291432520666476672330898490519143112524497354160674024115448646058805019008151722613262872134709025322700740962832993047949734418015818847196985816990563475660599639646689634728711939085583787489835290841951495871154123547657070442460128886123501763226534191379212539235152304965320878268126357276784159619510307569031267678554412327114533671151624874057852486724228341487306052938905622462761849307999599206986978957462547774448086809369132439508310852525358598175543821855189425294933838848534701708647100801690490226609745148611175982888613326332267005352928248382565353553408616225716159895050720126993653063405752256465693212009832540884679016463831248855914928775345614809978231741390706477038563907869455073669963995863364145317426094261712677124112996679142025201328383007219834845582349327074163293795521802055821295506646484157679236944714312630651431085904222720921369785068355013214197150375067453551843360570551164603325664823736254985388151381518075785374258285882410452882945101265718090642872284413454228662106257785049593642691799660036703173827129348516495213990144050384889598851208101525972340075578359729860136417923207503107748229108794261191326149088078907350278618197266630952643417328902351898376120584591361850209148029226344065161690639616391330102703045413896748718752634702235538446924634942871538642721570238049696613814404762710872213782589649551453956658363479378785534508577152427261190568200570959638710362855578249407854682636282434305801734445294440029566030793644824224430793125025671959431813006134863420879000760403551361048885822552478935877320550773433594873442773911284449682743814866857945112094641269540264824414831684660579694139346061832627461556911384989556309795011953872162618428969951115199842389945912010618300235803566917608932071831136257859468414552361212435396730737971805066913388610125017234733189293044765267420071703555261416068814388898515300180381717309201797059428184870500167390973334171110954114009609293407138550277125814310347617622812963898481633987541441463968114260701022871329720437849156689779340570837419072860930521770868384121409299227889854763138785171107381909469899179498961231380399774461840521739288557765524767739001444365287752255188771399358654460950636883395718233946078312771977494554852485978372233181670802026888867863433264450892982442878337533785738973596039179276163773037846214609517813106745901456862193809143614245701251841138544483586967994787063071954623723088955210289948172838102699137176185481068191705822484927398534181485678146533758718762392778623183914411923524411081475012961147136906308293406915375795537785233445750760470124856973541330266488736871462563212314152192563289541622008709928052674488269474935410156606670629752910537614585756146397713135218318073626300631389440954963792596826176969917979172213405503988079316347434779459312483609410026389311633524060196619330386781608331217677679205099180252813609957827385195681426075324015111486443692102875983973575700516410538331980293675552131809608357108893036990425834217806674509347125505257037286343073363602286358640811686789972825369389712698746444979253621162245352907216417113351245162403151929131541676442907032117881277198379295504394335736136890189380386690894344105925138687768962104399243853675390242266418658453304717546283248592013403921498355758321493503962116615256722585299255381634913750709835881909062373807922155141461519660647305424729659759304408637171698386937603994781710885901443732069981540437415163357328589067058773684436809292176446988318798877490007585351796826226677693875244126314919450848090987555290766096345877007612700470172884970821289215587271554356356921115780671768331489699576910634501991701661271802492558323772131453641026899715236304380377860307728620137202388350090301541005929140195786160011614261352202953511617168741067607796650973842708570584228675166782151177201902812180047264569097360805702513046056386834342028655055501548544751989138120509788353274794354570070313653766736100466335692205534268967671519870744997225596733642697143978710491256690448130809467976498038372372682855362879000842307501768532765283500935925140868594025606589152713807035707209056034537170182707299164994564739886679674982471713541047624723010533972371469293160337573314330629515241176493302503040982281204530419051840948241254043865280873095224690051860530794514777678543311320533272587807965932071260135307509734305363359576289659795473870560597388050686148712007626439448339829849705375436992138895969599931185259293314550989557365758041295802030797079886244203090420754418698600393585585437950805557893815838013728461870897735584350104009281312275419439621685300951325063148803753194165303459015410651483008868361894790629611983969124900893275960624986720232583517035705589086697289736721179704987060044677793107975798319606929815727240560277656015650545657718865491360218151893834836202675157089418946770436158611978601638645294359998573248984246379618637324012797462938932940604214941490317264755721304969565014505815650478428301123194916971957964384448092957806810203724556993642327347893472079878676686915412471195365222896222614713291244498352324217171390837162627458762020847896130397866956989511045291150757341893222878634585949386717314685610681520338788778373394661058975482329232560371113424307669857732632915523625069530720333969371651910534442246477759036431821649081328135962645405496551526143262314613812189459721503626897904880584097463776050643550942483886970003661060339156600493125565873453705461623462464593044234907696792551147020239444516694478655158921193915021596900908835855864731681345401959349881698245343293734777313503701992924212554946917348094884656567769211517667015430012315564364994376981458357757073082070863198921200492351734845608567022971198961159867663781570237938687352130962759825549128676592693017693831913294678438975653687872713536411666575181802626329613971246298832876283002642173788407031763033273352231537794927143136762386804744546531127289702586027454384105048600234840111082019732053778579629222862188541213318318264355371495251352856429032422981155928712995704406429194476965615493833134770564486848178992404397635306871064312556372506889622596096704844172344909375656316938745046200762719222238004393200902004791252398928003808202323252754578771474545518899053680609904201615961530398230828647048601736886477986259673254521697802247605661821736006294722489537635910318624994483712272852033183929311262259161664817761992702657800743129649352697098307861108996749332374956893859026847724352362512748768111041044526942225455475391098654455355615172658909142330617847223384620617881407939116995011741145752991054147107509408962239434865509371005890245193527205712313733983475757256245950996610000678126884901111102276689916568012816934238019876713132010452586610782858374064578174231905185605857044824270555663652717780347306616517444733612831945345411828554976307007272476624683570241638889557914279196531768452513053239087507279271925239974567067300544995019699187708471042221101525905625736750286382531424123588066487025633144807180576560768258802681157746692105546102729095142766739383785023043210345049280016964272645002412876128967075135511317530355696778327816449870777731124701022209175497282383422548946968230029541640928733701977750180166943544007052752504718705444906642746290564715969449063245986052695318788984779407537475372855668393983753669614023883899303539755906743757591665832527595434204271853013523040770909302797306787267531928854813903135963715044695170046328469653901332644689729164015963121639633456229631638087583361129338443943348266474281090985928667871182048424204675764667872612613716870852339204190759048155466593044029120639064620309813665610912582423666088896939635507417800560858622187971647100528333883883132337669298810211827440237797651915794647203567344349774447199615158417939060786575275333540353423475126548093247871520023808410853391665858642244148418989324344434792060919663525220543286880374472496952620711947162386866878639106559317194916773208235648518653609495642001199027506056820484554286625853058003809110106679228365018014522388098710514955543013239135483865696064413538009533992139900205322416101910746232519946310126501128192494590041098452935384914229139586643172181686885364434526879210020330097653602975923376923541909821627957180808021043880620700058866218431340351373952135354418195874089760899029428715353806682038051006595686105218391234990230270832339216152831727311689246878565128718701699527409140318507265692870619215092804420782143760005599407589167761543194193795359610785212947990100770741240444573104002806952171827746030329657071570668756207147871143318489801783984860841185389746582890727231907528222009466725523992592616348250299565755266101890579043175668758351657231209101295267778864310550333157359935570582039545624053391617647727833307758808128246771287837072299466455699983149894784030633294579391031509088201351916201772628182144290727941367423348193488248905895690385265691648029704929220125708942748322491929073515363659206595475007167249615615507413665024553554343682038232803037684708916823445555496031809013313153094239307027640749301036142768165203048838567955350067490127885184014237017493210443836708674458565854132989258857003901509214779500309155886487115461229004851958486650737865797711857546952691980720204762837860163571372740607001439473729783510608631512570851801987826375464802974328640470274207587281966729306012319761107111565232308830803199785736770643639628880352839901256104418006222710207085413083668149826536338446515304911009236835626284221082195589538915029057221525871471206986873240603339030668936584499282722321004745823808419914228967067688745216310043277006665785602865107670763424417053329052702756690147864470932881267479073793056732896397470209147283362294490676818787845327891366333336409642564976714898095649892460474227282012762225551608733969724108996687820505115448023665871209497590639489961304233715643343499572569225827195537670434654241710979697458956431452914941736970804981799079930753488737406241532242796970913180443180962177890815234108678557774786470906288670185912869807149119561201338008077044500979911261318635852461893958392770405722108451986476682790116687067601396903365903712294643540289871033149917845322903559599975295983142223898266657544909314019026094052029159941999300347696632772504781992038826456116138795258353723680363611105696936395981556130084402663316337708015315356213257391428659485493650781838299648678959447860250606181108052642872925747120116604311734108800480448139609879272916591831074844677780759336525771264470230870624809247358062891862039921124837564376411472936074334551076260477885507534297618722164572659235059720123962130948362238467200989899740514168548882240703677972117024552609148459577873719657167698154418352257447242519401527279000145379832225214232981728194234658569993928038316886674852832686824931500874961276461143566601590650765150781112482714201579952745092182024445015768255754077890519591165667547143614333971243170233006545276789762849933816317627084936106026648122307949759455063349980948889627668639431267206022852091594907370593305997225929145566453063542634711275570826888786450798091888322481808916094137813210376534241436451647805598437341646895009516196198387664014546677705899366705511888510952127589468576426863793224746760520728624839900685967550180699900469575250145982002236084059868456200298172607316686805823559578606064335791813131647688828452188065982675159289590681811900608068371829429668980837690829286978894513425526855662310196325312217072814923754967892095672748315573616622350056480306154912679782999675668977863681161912161420703885067434396244882670010551088742873132802953599842465451786970162403493835771744376123717441974197408727173159468944124263788549413550013467375654787775051912998566561766274393645157416668003953181152155455760138772565153257322638465239656333153743535426029957613319058646136574883259662329234014651206499408107927929382320004581653490511252312099086536992526167223191087959302406682375133075775927855219912731524502778423991796549053219638023721154937152707129888227456846549863956631880982248338568826258833134520525617436948524031931122900892767024908380477662601807089851072488269009445579008457891498060155452524773454181671764995177053840091286235623049101334069729176015874578451572944476793105766880755259227307294813752961319257476959860352918290839413522755985102210295184889514285031298852752302600502998673484768819722619783950877924217927173041966971568055108545392718865380877386727385344835250933921783721628163144341867058564892448033960634257317250453026993449268799854023417389694344410073181541849422692618039063512622576721458510879376190743782646698521319752360076539412259920877572873137954598215504549940674736899666020853620851963381573213281755641135856543164803325881106393986564592731548908387472008947711997436769641614281113081293309639343256517996996280193230091336321006500321604272201821054083381642801363043869499081471403737020659448972939010982462893016804840664224518465299120607641821661987063158129804772068088853934286472721580952247562200641412661769609321234630027490285928122094241073293725527550967684269372445585985750904593992847024345463318797250966520450823420889623336109666994879917960562694778125921431838278289063560856871165107568059483841773726464680150095620511782086781098547745533725902688894467163107824730780994136286989686367208227261471425867050908543479528762463961356801264851214691996844329325856773215739252053809494142590416524917535305221489447858750609679436167591602498521958080557073939004438347915577937023840276605085019200762654399563821748141510044239731249371276864542532046869946556576948396876181406062755830742315469128106342394603600232722668864660769233879783911003756857255818154843277612226957868215391051625360229770161835465489600914325397096044383373761199446041298545878570639897355642686958141578390405047079676054514857505617165721897535616472874181939118554014246726961963542349924255108110104353136920374249805351475416115508898416118260548026185483525695802914506430413185117236712854142273649609275485311919077504714070822226284248317066210388283061010042971710262173978538309475814564719464656610084668584103185815396083669075583941676415086765488190414168670370357214086485100746935167183961098848794660984067415624591688426509493822292732760539964492538186288877237549207331052937574535223881599377880311710350912539121563523528816171726374613481210824889995258292183585142789169489858726732392301632463269810779138051235009594133580833991669813861702649636106870443112263933165613247764342064437108510549904292383637478670426003876502246743859992781042178451534624279517602978598967099776497337827917375013616999058991923281768005025484958099327095119115779827689028255627015734829538739601619807455881305752941407491426670680251846561971215239694121619991063849864762842008982227438500702117102725054846774093925486522282097077980835628526559906095864257598346860149625623907499869153305560958457406792917094546607778404806966417651373240090009799439005018160556221145209090136636325909393780412473980636451215453247922626587584634656117208340453647787751153205732076065826866525094638868330371813826847769967110146549754332924122397705415815818835130234981182137177802245262784018041150370448270504184984192976457933474852907183093206384687859613191787873953059417395307931259614479813167718765986168377290632205077900943352437925598535330292276725672457188848776357508467940363116382621929489665081629140511459040899702503452695751606384331594222392771129348079781412623943074530153811960585696298499232506771778583638585771015934730656436406307516447788004650501289452022243352011086989575177192856627759532404661378796851966828890264362939133016499907589502537637300852703874457302206007438222676692188822075365132750232730730090723908134080292577356385591467089658462390618260230807690431266622164264239502358139866121397780018852824637821369270993036896581739687751651894862450997333940461359213605856365870080877242877589232849167362177817123642065639698210698496665407439916241571146118388381915743007261773703574410365987424224218989692025191484207366805588530076720922765091082903650621166999122311761524723942704680978125289175720153403408220851466637001620684320593635928363828177469457879103185231849466898702715260086597133782585493568262075699961733479009301887859504897735518816807339594420294724329274312748538167934834482486180231194279846963252803929299758597579886073106896626772412591669778826351109951585494794511186188588537618263470791271090662666002378339386619023504205776095281301399165086589846343470462120227293894213593450688145450574986501946794658522166915848891255139349656360967504484837670964141085220321641161335064557973801151027743744467811625255874828663121661207998997904339126487910379288293099977079719733563418547821105087550664619972048996497729675136865768753066653763755282503969917853070171065110928641314883357559375059367366698049891326867564004025146206357015954922007165108992449740039382012441464439509312229038355764352299175083109916073344804581850574119966920826312821477321699345105129685824327703115850239567028186060748722554820214499087281122121336761296153146121693654093626484230460201207378078831337586866234655644629514271574177776657502247074599991019126084938063760407812137432778694534092816135065178602847942766340321293433971186564478004361041697105257201438114850167940599913392162994670163035662534401791177076929002887497394057646175164558914956132900707113191732391714091427050197859742587070769072883348364217672377925646474760843651589011215844013097383244071716514834734831562293174868563559036027495903164949027681180939602261192957333408063776519906828782546715702363481120803224740890378349359092878112624669589321431343440004520891764125193247555667427263160297869884533873330248066196438502888007305734656258823309406252540596896665984539536611633044880744324581483656268135160339372223227572631251150059455010058603181963439365007262606695554132154681623028594033460341169279127052794361175479536459859248853422495377853036974610426454159482610398195106516127970167219512584657144650119437559168618928939883452394106733606466598998593023617474846464390281863627076350887023363432490545698983410128052614887658453351102044487476095551661798122198988026370315818259063606912776245156704723410891633691351392336629743085855168872737189358329967656259889293574938900318654051028741971798612053594750151748139213999496562673183324302775001458219312750704137224938621565617047502847459700640679551562659784315245911634938376683091935216335145738866173389129500536094178923335747332446158176125306009696687640388193350091023572537555563386756148674388681624474757087344401673850591020976811939751547406504992794098598750330748010567063418711456321652224811648411368564568967362990840828989105140242466925723547713031159417150241524701710957291545799630100512926556957394557036540765999341394615430154246108450423495015133704354173456266539295638136948729892649989169474585503638889032833496816428459548840127246387530010698489717887865766161036839867735458029450447046921527296889200974577171667253313917103588195608422534322584026407137585625594637481439944035806455903690748777056389075627586790201507861324309170362454641436264340926615911162083127911770273214164973968348123367083410586484664561592671969837448104597669022996362156834137170601759146883685756139470281886173548675796616811012619353731117227448387937711737103473612443544721423733872887820949370465954339988584019303928949752091730885903796103662172386371171166237923450343057069887046900366284662454228523757001208573087764386440756099889290795749151933534472386448166766768038999704545986649745106429202618180644592728655154931960160256333495022226772666472354066130914703325297910396230726429857664055573844701231738103828034927941968430714235196038027145226335141683996790018805726938777438677551546893318951845631313786546328243403501587046080900028335034276708715659870183014893550658246427117176019618114998798950933532429227817028707236456780505164971405566941310254445224056147670275162325156980005685892203243069573139423111948260162343431280987569693790257262663536919160989059793441008075278897449922087433196910785162078239744996151220000505791006630566469836577065352790946664354362374901285655473413744354435713273302615959767964930203362126039363737499239458473634609764795778469259136182548616819480155991834393957620606258050969464799980964328555976132533676123732063839374352963237389390840515169130031678891371623979691374373205411174535495666620962635055370491371035236022790516033411309619444730509923609035330968372361165901589780042060529016938929401675826308182901814922383175735339108449533543308784337590673667100890796883895008928832186360267411857991181996772744857846933616811034441724610053495142149220148548499732014607195372217323979824506246901211578932685866874144301620108878409695330733513911779966108376748008299805121575525341719392049882170805922915165035507823633906896585239553686571269910046166258969562863225837557198939248801892728146509831732278120289710363346296000134554262667437400298182346059858978127159834493474981861052677748360453510376711758878626592588923687009787154927945242897141822350865623141539636824476941812608416421435006123861929517698665334958669800200090820405885006324214359998789346920302111701343300387202948760201983582432003200561630747440115466487406645826626225667422751840554219208093681564925140363776587898455484988181899815278677885894694295834338603755639629027274041780044706418846954317570712071590613597470811079009369855620992070844891570660999929476579582964432509984313974283966589149205885429659619015083385059625970892895102765214242488011242362291595963254036069218142035508576481180775007127834220876203233565850983242553842419832829864779480630372374320623449020437794631344765488146101612025581269964406414466467717811339957537726150886951672911117089000170631742060662171626744408621639548084790675193480147653559650810425663876999486674483853798150659177530054430270324820313019263896638450868283886832330378007962027578651870791732971667611637222266288247363760564178348583873624364421410903007017116558935329941199736130655471926349330572235370110652273041594858931725258668389837772298047547364805254057118757861657286438124570124467893241175284829239294461204779129573262924341585991369235783583271860386058312176079258462302130920069179458357757635915824237152914354291375529448250295466473126692361256584630940328644681692586448623271934711215363321239692664587592949237570189635878176490769351579615787503264551419348152177712350694324169069388530508050630059071643488897316256376080653763443803175129720684629710024467730249832163145244677944839202854682923744535195080042682754917367604771383001758272128098702496967464424212034574491013020206883491639516989406969838371917028483056789214591150483671067093311833963638124682412574239248194129836802182786485032380338993432092711075064692261876693724843755085448096385689192063486432423708838312536398752790526563277030775445251672645283656425360695341707058980121279054450905907139711745526252159514234153234237077408165029791164586032796193932715049683075652579578197330747736860014600479282669807882979107590799708677216503359174502372330190982484321540123310435364733726500519549922476674821602389642226238196348811922886707987934605026805556164035679915667050592241200014617500836808545050121244776873243409529029952421951062587422384317677322256865754770964669012445759051770548252208450316242682444435331144921014021238568390777516401283040375527925164939116857095738634921262613324626886380111869765849112999342036154435668527213959682757230872363823428774746410049255978368821483395040289904061831325457712051850375246064814121253274094741065379919357322630501302032663870672398666950099307859541075748687076943139237816299961606204743899874651862371411826424130467554503150493026034734115775289147049831707866703032764756735828551911367430751594856601612238226203663949796594934521297419647748618339517052694622638367698219499068599963569748729967393320324476382694341111665598083378623208022464079064563470286264834853371967008118997470369272359901069089117446801802835891720370333425111770187059792193383283381647328638599073117655682423786684206530862382481446395074458960127348216481798767028929161832769056826923428874168295753841909546448858431939386320913663058116206853920546054479039512862070706132984973910350932329572764661499002382288455253073005113324819557853504404808745525939083071487237805381038961496071094881792007480359582760560136748497010808426560448402946999217800974701100152650271552283097637189982319903092842093424300415201189234935783942639113037897948852425475813363616991661679427121302859097078237561645345209883964739898456291381966939330808584286954163635626416661478619067740171289027029316462143208696971645330608071398981941779147015145960833441829802358740396906819959714899894516681904015781564814468231282505870984261106407540451299888828084898781607887353483854187981434230731053631635232652499176479062896581561588616828542621957736994057695787340606737210125613518069953018891672178595628111015561980341441750849724400478097072421125310833837182505425785200589459219565016458941085841739005156282547289542195895230266414309722371531161989467087316146200599623698883094010650753226971771931557391482686304169502080353622602538620709444060008127802794882693566315938267428495870533806178644026027248767706809965090841607916616510640376731280107878845380380154637768129990968827496799231181182184241799742498583232071205056759151972215947764543413539493234287823262695651051847352787121312883926879849824932663693657136260988126646254994115376826367064376715890283562838726604760509298602576574055142646741819693857172687952759545765897994260264993072176684654439929063604083581745073804516106637494776383377058956943769762504568158951405064488042466310522647222369908222207267806698569447120551348077702221986155074103220442945150052236920397358160788152218119797945194806310153176744004434345782355327435292346131395929281578712989856087150869362197715184057867125247996026912791220604974311527540899640409125168721830505486985814884988174772988166808890470201358830439859885109757947579298195110045228914379620019504818068936368419911197248420482210971020264261048213646619824322728711462769189873121485375794681949347604179456995468337423210444552325647519325869290089676211009463490501511207761930370537913343871811259663794045410591745560057783697171610460667174985149344610901963148803277288739402273259458345294899509449427660465054557557854611405386638211379868131032308595090234004414981419873518211624830674502821764290458475183371306638200097424155135981040675453064542320612690828789011185193266235983692773721866589071730578533750433152908603236462190686209714585029388995727489911680840741447517286385996131019612161279542143116643148941470970701672117552741749583489329882191441644906177267165089796094673157722029081759931038483909510645770615131123428172529669852868035794737906233026971792529274295430530604118192181385722987094126849541646106217863363690425241331003051001335097644838772141193444614126162382404806905503906576511387895098263869110394173856477090683965185284962640734886506435068125869656253575690304277770937900096035166429689652489070620237879369107588990300015487843579894875945020817243080938284984702649684158956261244973918593586149810922797654702603378299602929223639941738029402078656947507285004786829771182501488760266408367323699226679947698424521660980379910890694785894743528993725092882615173557275979813346084287499334236804467543258708368243288023610114219674309036256341287849418634408803886772298686372324181068133240534031087691939789169709172375799682368690641890998774198288622800040912889367481052835681016403116813747071973237972962250588661113387025797750639406303724847434356495579260148212855161191211142760025292413838375937262063096957371424425810743951467859249127572807033968720062685626264254151664957559740750874415709628798796228155610764821125309933458877616987883473142523609416496558510371520584642983683349907768378270247973211720869284649396554960372890057526312501043854766313566673824227808635706879324043594106799544447944939612708770464178961336267510566506842614646923940316933790562712910202530263222016630029406129470372634538942221443336372697607938892174412121866328361760969194024637738988488607771284351099564505803324406050303387109556017399697060278134046558377309774711091356027846696423207998285629990659422796804227337825560878389076228581757638941458361890170905326249409271138989521305463057055843004774993250107653806876868065445434458684027025747155406092922987718305045598027423506615683114276670546079491824871336350518619011771374789080751296688691611920118026045460085364608461412044413998307224893688615769377966918494919821830862514963276919278148524454170035156298696454889550696419114342517294391698351435634638911776442291149293505714433367893178860152284560171547981397195933184415557667103217441699682044449433202411861974357144566357443386443639388910588428238617219052049759803663321238210916338705886155440693501707500971813329382124803156050139531760668205080598722981930121821884986051619321270459540184268314940595211315614905030298174881592114801583817933049073672388265109886519250987890510576386926552122925691980069578016882380020132013274563697722589642914856570737643952725698083797654123685852452188249291034175736550314894568256294333408444527150382543750799296156954380685664129147257134431654989390988652966808581776301968308859662225262329701456472306309243760354837414186819798871836048748664575647427788517435923767633580546471295337253161271962327875799952171794124379337734581682178304872586872305813011915843273741950715458829461825296455870955171517412501300389500899396479843122957155890308923540670713582090815530324320081145411389738182112525807592607592206084179038349435911766193660873376362345420701451995063544219576030462470273872619997997397591178577008723582784316705829216567932032959302933541836826957684641405568723648964395798760252512759995656961480395096615882421310139176368283461692581212178875551137548745846433741638679126528294190797491611059108777850357416709005593668756697904463322465025634845605165138345712297558692575616364358555195956648791155568049981825985701389344459290764190717961257975724400170974373613156241881436940666381441904746560782352589350057348002503566361231797897011060838301581314694616842066876471174448760364164967777101803517287938900420893119448186798493454574750767397648655179747195600610171277393265120621151562611093042012408885683872705835076658638859332453644755158050575750041228748525801020325918389827183822487023489374300488030141995849577565441408350485067196487328409694779212936060253212866243993169900527812326230082534244126082700945162218089377732523352171514853427328526313195125193581326795670329751623628764650194055790443608838832326288214294250645999127427917943479812788810525745140233897049330582700586304609595369647553784043440104908581052704837230298383521397894803068444721075081907619649186270729942079488578972629451289703443369368384734193351811779431063581339857827778941715320346679434738338451838448064278931895319548545919462224306982320886204615563199640952854398141124000619311505657275288616971376476317352656646399444145393657959570421581468217413720939626917179311441633504100272017285306991217914261222779602264290591908906098503678019985663591073448756819997489574181907225589243684330255803737765023948210086615157894122496782746618160689679569145577252033680958972465352725494270288449204991522705893189588478840873110756792814184489192480079765114702408365132170856613084143711826696440255129007771025184073866660362349775590631057733330003675827479304463501274521614016183504245601682502896813255962438707380058588965100796695079937194818507130970746319156591665115753797333835065302230576967313501262541527232211761253315607873681055328237747307709103471976621077864603661796790604153601045027670513831330608810006308690848165394939871863410964353725588571652172489783294020554375978984805836846891661353765284510510889081515840733383347600703954112725898599879311069270758793864626354926750122423564695832827882207012372282518146081544390195305663803961704573200695481994015295190638111537239571293652619986366368583659346352491360021200444674099275997266483064477781223594278407760892853821723650478613617964621404302892029178711544686345713542806369853683313933179996831139663834496116833176622160212053774379147719035960483719724103791375492892947183666959138882523666457220282215216141834867811025738695832900734377728843964468642705536719046771019681272329564298949689761930209958735611804412505297525517085816258473030434223459917097302690008051104873883523075073498415021022807564880456726860051144348695658399096084665494193565135454375087148378323192780575881045807508085108597672271281888697113940049379350539969422176779336287840350481655169713520517664562477396295422073040628813755957611873496713355690750218302318764000505795216078429266121264214095288337973602480371984919548867142884674571758035445832265759766893652820299511027190763834567936643666606393609237412079942504186024121377877211842367776464763950000537362208291934341613368315024408072619590094007744493352682400295586884316925334558193658700659585952488165926049720107420129565884911833633618048715226708645030721155678172641843148952066251123573211557763691033012392234213888261156725093002369456244812861457236571777089839807735770656759737788531541505712834673452956573369112130528428848308924458477308002459252693081804256004333510249309106994809959186990462723475631664689876242166296294571456481364732827198193300576852860123743729381215607772591065598252068439045048765144656517379735448096279359375569040804617332813795804933970396279401794920455735737835432337877754560145065138787989001631433208287604260474684553049451787655644816660957021199364905469489335271982894436917201790691783800860073656516531463025274991597549279659354023442535619927279434320483068854601132994543862679000140326734886508325598791655661721273447366523268963875306932828150767249428885551763170785231626587587443307578084879592655579232658606643213144389615647739516525337925050245129762724101155186600835234101458715177132277167647742195258243268502484092103165127441909089810026926137297584619925842560271860670253187452729119842755084144892157271079162067154196147462369492130792711547391841865345725760818938057017210981480853041574997353511994565843214522369264601243615201161014817050888902947810574305438193522829795515946557719608138480609340099785736539807516759367190119143650987747171584742754123404764284921264388559300181314302626011174210382754002144384447571688571574253889845728975271857568800998628345700472285430608026274265111444001122839260483186565428540300628640711040425364070921253334667890764257775482989903923627966214793455505474517534210256386971024129844740474813676196888041024615987612585093586472413411110309694060191322019313656380516838843919198357155851916828525543167794624753727056607189404261760162892639078907322332488015286090422062807122489018118152324592361118259316375361993828478315717529441858583557338148036950482803348715846834373647036251682613118749867108334935010676388176059838350991743099239609420434453965978597647616629912389142523430431485177643478255020321737376124522044039409138684458550607198021647409769212538402899638148607546691997659690611859463996267036162599382230906671506102710134047589432732108873990207063803816767416643421972153779300673207329565613376959812672310210574419777408789684150174501351611161592550868298009529639984217638761167307553163617758687910791197211724751135939100379628763292547591723485632984579536150511673558888764510601215710314911500657714614443038570969090417972018644411111344630194658411728465456120336174423561368044381160795480060291774389562677645763728388998097047630979197182269422864993055669072904140445675549293219046223796393723535690544642807630762784774785777161989007439032475197637673960290613346648946817027443232800366687173574209942166571926149216298808677765753269111890348668786651337094897000058058465689215529478960384322043033500429230827123858841902505190581040342613220224336907927085828012300241836673817123326923440545557697017696877662534547126170083708220615239495013995322976948747746648969511394351828636574826270296308684610793069542053375142154837947089785494296419575608218551435390466957472429286010781326726070839790735760013661683134416887319663713264532990885855691297177769834406952329566594282557075106769464652992275056170859820334682555019968717938618829715600646429599875669239460599002865903228096636883082027779171904821310042908629190704101255880982602280175824975524568510753766241126767206727529362055162361932767567126869161023007801386593287555766106049577007764842409702745289680890186127234396927882266308822833895877451573212760231800694421191223232529269999946884895372959726854577345697265013728099464916067875706206376518575104856100344903270934397514775469514823375682832336761190599190879754067219235007775219157826108211368312426280267123488030942387122056228100823039901536329914441097739117203995292894647320681094623087052345743111094272/9502481674695341188780792376334379145927500130548141474947087037800933368439754382825248029876425609187254238401349932205752395190217474257147452808367272354215808362748844951291641387082454280151260534501667367525711949615529387039261613627673702310736836851474241526646200165958849785078906299797158831103302631134695179958740848426733929122873404790360578594995551373351164892150636836697439908589998560414242950785334447901421446108421403589660458923018048464854511893645859034126163657732148061454077873198482121818042913560171889358128993214977074595794125297437690475706823190730351433638148634962419665449552229402114648420426733197546204891351415258241481489646339136081039545114317253128608867553054306176043961664116468051859098519105516453995177141298819011523170647109573459540255644166685185966732708781343929648215433272965093014710388284100282856721881500280562838431606860000846101059683729050254974943034225188436416606741793872028374075725587011612922120158902887080490525823386542618832487766218607061356239801558124538243565333059000996165306179800849932391702768979722794961680470780029907314639835552019635746632869778602623471661567831190480289669076900397972331137678447113018433698775714389098459122985617444916236099112357055671682665520229172146096158342404875962893256207473280647289245594398571408214917549155505308394969264976366817230829596761296421240364679361403621857397797304570724173742292912821158618504868062870483099717171641254368342330221999895408337163131226832469517878205799036209877732098840555622047246514441047455867800854999278131054985901595827481033304863318244555105451954384367337579674879906604405983804547081204668071614221888559786094818433193838302290433859237926548744319441270831730009916432754916108642287607049463692078114350998811145082196336704410042016394803394121689085331495760137669183736559784538844728133713788376948429869470226047546205707818740543955741930335691209623153897424762155791276882972805232790025495376890107128975476011342608374815265658723131594031288407290559320175882780928683965213949801845843520992943197953789552524916816897257556946455348602193593769929498855674380511366576957010137786713207851293931722807546858264200552585475869732925838804916797423752831760534047362323939369966433804448790463856864050151073274872958994796645778180906756720130121158228464701246043751388129156395910230804338073981120149277090668700184010519754813760304701172752009832343126676982947338104092748562342333533188364626265614635572291565540703180093879127856442488841979982329505771691143927176319910045023138444651494562870384927884953454383518950194650399226803304090598093226013196221260138579924221076127676779462969056422022127512715847945730850528206080533102858155443989455681806446261252498535077407755811825056515962246658607209869978055058819766838931396704970533227236851772425083441742869913538474317185576704648241713436839947906513976583102537924761883120477953790656946506554841129521547977300928471704548281188918161674891136371438797609241723559646144408386757321497820287440498125175717245800388813953130192398906077879461536009878529470558094551125262723247243965676761122047825775825882260320097090401822164404292167308681532268365349384986769015311380744328294974320068179138825684251979346540696155604336961369732597932350108765408856303555488648308211499999201141553567482150095813730490369897975849913110886972788223390343324086172040682420409041139065133263944775559870336671834847351354175323517981979963648066964484260451130854677635338066644096893151870774018130758310831984901168879287555288227339569846567294300202583759628089101478454889372729666200574778254971132599132757591958167541602808550338942696591124520977221560606226508755083948171561400338220032843102459127049076197545893261852303498458117253497377389735750487784287151759973125584313775877400572316982719357058239765405012498347988921325722123294504560235955984921332539199946470794295950966176668293702633470688737160492244164025549257633003281462647070017271700026066669742982620285733877923424843234523280658489769013756681639730611262710802842205839004645365463505569871117006044529853904753276185429650513451532373911718330727606222076367244985117771790723773196616291360572748351557535898825813577964583535767858071836464370206050363936873000264553114493513093331627845871102566893108473293058095182645996989220338479071161872000257639106840338820636001440407182770202256270485268056727208914370740343486971337697681391240617207210991286553983268576521674874485754202985691615704821994226262922881236099505463670353704480669087487605119238891834023409478364236890333691956166302515428099120270196683921016110496186549294129506698326742365845152717722414840079169040248893400243579736681490062717840041980412239146078989401322507925820738425989059403660428070473560774789557012631885538645321379951542433749742272491187166745640004746445934441872187436380690560863934188594830740747942387215538239114993547496815516962879823082592825369280886989717554367354117534853158554179536199644626071870468184935707633604582388152126454009057558876055877420852052413935924540366381765728761281284025705219122241266363971488286375344614168962210507784017608301012469263640444130453561204841235355306402714065507223809881602722000815348101659337962130295371651230799163884478935250699155079472820741870179380604540651200275411940151503545354845054397520873830665489031445329288341751362209020620867183563692568231846672304610372189069940607240470228792437831476910194431703309534683658781143187792486285837807915074717249319017262204176512726546155950797181838217978563881798739777765533372145477537002254064463509783214981753317333347430571259866879015163622547091611972257719527743877867247701556950838245956667516020817992056725252989926127475921948688539766862772556538959445107885879209611664650066188658762473145688609378946096146094858612035431108796497057532304003366735350495646257414375455284264242694042739002357280375291609575132563268297879787371046066315092237143003469430877779920645804402706475049239040347484957256214557146902285568215194510375306781454616884169477743354105911793039720023695266273144479250551459132139572924289884850316168726072733118332581077734925862551566532016280436633769577329907997158361243059496448683203041362200005424442518936258499153755270058981713660012840212159174310668258895994445480546930891358695878743733165401983409278970522289120410591233801778321666302524262502041468978893424835239394458898299025987108408756241981381438430538487131910681964910239104257281004112639034435646477572668791697374467017699887660992321063327204449096966593396570706953792918185014730101761187529846483068535168042486380628789291894794043640965742024461227301964268842296207285903083894558935779598360490205672700195218271769171888684929428229451279693484243249959789184628192039809734567156083521808569328720709977286031665384369172709927464782111084481464077578969993318424706182009951530924496232685552156642297294854136631116013358121817989864434200476281378724401254650268527595674651156379286467815318107933852409800814916327049469384741076877797045364605858013666841353007565618189071922477329633285931617560314891062179687186013297554980445636108863756493124951190110971974128995894460085297077015694813189836715172664310144101069243672468065534634194503757129859706176438277657799673857588780901525772017988488074037400156133284702029241273329715667235883001572112701349457215832154397365786561305896943986213016448305022914609206552238005755569833952017277596208365133767516061874255125281215562453493162958462906712844010981753327654678036077199232433305007914002778666555983164925213093735708864606631998202750373689011943537024957293722048267456516193948279947479707789352798261984517842813120382512601244983287359054605662322057422745031184068809217972721827610246653066225320354658264058061014454839633426671458195443925209982559884067110294293695531784817075903431044360592900986133620841589755499279096124893102625848804738671835732880716850113477637693102817232564940005508013967874719515585137919530696737592506957628380309455360926917483777897809057172945451002365245514764249726631300347421079689364557524682393180611554811879428510658283308727883866236353409952883439949656157170386626230594170598999734149374182901488230125141695466568000733375839462073750210630699640776477369690381107175497251429335162228908590366220591614973076754616537739866995810692885928155196189514850150029666296678580438691381936985758162985914791396069959063591477585004793964667001730538161987191381678447329817078055552773373606951617504846138641016559144664105349987944585226913001799683798280682216349516511846781595293609558027572697927280925078884511174964425118998147772533000137405516481067448042828319198644471620948261032708965221201784822714843665668991443902554853187367829810514346991915803819086352206696092762280901312864423480766010614738807742503871870788324845239558419593110841062688937233770834154620550525515659065278987016907590403998182596400939417674106310260790651451998947310304173177376580310292775368674759859858910953447536997696206068466765178200748953349963370534897868994203351586370163685334143252509028363593141073110193532138110314098703986623871912938100972647499860159284987679006787909350208722568596888790112440174331243266227976090091321599470611484364740117185447265505982169743476160041235198023791786023211059240146057770275211999879538292901380664769801804447342545586367417701499878740390383427394025227494317032118055060831158328024723665051959019641314697901147459605672334316813169591129941597734838700276724505548405623471033380049357715240400302533821891236336220581530144061030230565743693436537522272330863602905277785906186249790176870028534778763254110374027857562731275265847804750333768629426884765188231230123270902355230099121181625756782045265672202217631977781542456332425459774028611435310960956250097096290301364173617125849531198095137939968021831162265242917211364767153675089593957369313180486987423058985739925204800748374583113819853545486755482798726632136421428308883690319139346224750140289973897198232046191686636096165196942108637226970636159259915070945183678014110741891121070508362250417733986245754241100593432089582756308346389342487206051878468866871278480924314217135468438195602032497248453185587512666836024105061586065298585800958155256221393648496084208842481068060012165093164716997818434606232613793353753952321037825916473820602763548830252135736662724861315716381932811064089475093395017367255275646738483627489687901092450842202842569528671445306543812397773049051877415848440288154117570898660711026269189065009227388439839457324255184469527371534752520779214896157307985332637802352867717452069178748884701871859931732327680707947653988357440984759008252304967513976585180642406123109908955354008045783038091200179174331614318853584352641577348490342211794068919770314893299201561175735121762046223828329903099447289400740466872393925288899125683362768986590651187658943994625477682296596797736934373921792705230214515537059787143058845771803264925444713102699745246653755843777517646097434482231272172903349001855629790354255028370433504030284910701843184514148965996799085429229066511589910873966061680823746721590450732014447992700168827671268441433490850252647490214803533374383045811482060498711118127915587696541536657157343976855307483239206799751172151503991623537453185512325555661083664566507078336605128347199551743561795051319301156449135392465296765101996885315863374120816843061007175023322358259561215504224303478551322934362979265024590493584298960699750941165223992146273307127069124886497993636446742488619005469362557105173865048834927644848009116526833736209929471238310610166729468819162272661123236738831204201072002372049822053676068263542478346224085064930420602578835859458920637730294429617727293462676481360455156715945915422097651769423854064639483027357636253988063511783009312439667723591122220510172478292062404641064511800236904836549415656726495093741441592126472118589322434597500540703500252452239818854445188598016846653810610889019747736686626291983695786316825725906366703267157696265955688279008226431026170975469085955566222858648677982568440556614019314968547137710358919448736023861659289128322327709977069252099821309579266061050963791892860928762931271251974908656727719805927068138960683815042909648303309566844983959926479303448348068254738008849845235955822090714720807644521129205655875870887115793361895957032865361425151958396402496240012566377166688253869234243084335950503512814561063109806408115142838953487512875562406685683173763845505636701827300656621526478272818094128058005430849485715077328590500678709079076218051869777032030853394780346550508957400677828466279016154457953179421840102883031457069223351029267036672749873263273358617550507519863526917473540439407892927448230307076829396799307822499796119435373025332057288566612872479551980848974590772000518991765178116717237305460452104004618475151531997822810792509897879712731574096449935271352966844031978750049734725100033716916563342399582638922037896222800387655517275085875353571462220344151472598593290011378456275857893977731289573217930008657792048114803527637960180004886901052652753495757438739443832559366088108514315923900819672980894653634502472122248346685559980376108704477816000389308463743652142317503081624312322246810356828350729307812768357132570664718217098914016410102205405560672246944012346402564279718827967780407939907017308304214414647579020277790754570769342203268163767501480617190492118023757415659305498849418342394776587542064977997464491911359606024399600302370060562676547439068303747256778722942769979991779757181412891683257159520248777199948053619473364342084199592760953049320472149145196883799427068355531602520660115936268855035603831324644993731885709092265899241070081674402491335340694767841346135644855319372588816427242325319452711796968975641594441022159325530338004952642747054471609795358689052976124531133269177702558964164387612929057613360869087971231667191599802980628696558163805341500879577389453732174266493678113907785834195085986553864843083904782180593877979258596522981978207025433670703474016560957737221396014179347561616150445169091302637916147773295320990316536591389098183844730696527550800860080332009989227442807495906711807102370431792870323199199899590884473927158535470496168891241012500522157748627696023253520396867177348184064094896818924084878114415592712047024634411585631156557041465029642441416813934464640252930054215338297505550508465777165234667977022027037833197361595624790431558442742576572638555494456912729796436798258677002081298131885322306755380782123030480408710951365154070166575043226947419609409437017833348188388629963052927893315987762917669519613038137097735280446965401774207741098376339145144147687926241937309629592174050102968099106688273336081741246066643568605197497225676258619353586473244783925559400116107901320647770528099335706758219824069617187365535449567170489744875186446771863889606142985687069192364416985160049978602145807742147654335267791944237670211875800555082030800675068265558370444248150175642599177730392010840342591497105714329318663469203443994519894292194330196620359567045363185440888288046287413734312004004054058030872483191582955025849099034292288096098215941379159329060648829109007388213645162222080647073776686893416794912738480755742480136471193634974401341736451403671149602093251483886634945501531303414949567582550215882673620622932619825072011180521714871937354707378476492374294597295799429577174886950426281341765903609920114504596416869466449275758077405136418662529008397437076854515982630046788562775564465866491957343649923088807436794745399371802538545662997778106596093421980294177221449613771820891336341931221773533665415704586917642660545910488189148191991025269301278489696450199050597852803871782921017642008069200573020055335844722167273518677046392136225711038263435675951282781955249088258990556459625827079915204068152850991505472885051528134312975624449923873675589722328049636581854993328435458999941512662936005407825183637466761885903597929217614795834061754601297420961034072522086205734166660302762739892821039457004626000574341869899661510379608253799516391508487296553563266010301943928210556675242852182159503273909300004777138153661597220452654571477777246321626871151705050348938756254836108687794023128816343784059742156296469686133568268651694634163017969090353719112390322547743562323943654349929224811113057389777214797826638520057221475110655121388680331332882662857541117407645641598497984322184242472226300429451792458415732442779880262159970761113200760974771913203106351529553280650615618035088770194939385054387301514038034643643952917423571767813581801263913476166858509014695342534667385275648105581042837073530675918997050993503583197785893805520363863056530584774267383749497975298626608581695467606621473450115176312140257867707847169466175755933136363459453256661655027181500600365248230657274722508140499727218998087359866662970062078885906735425899648757883299489897211160029219308116109629972796618483809938710375678385958582062671791704383034245308883327058953889999583832745214181268975772042618677533884070909942491755386847168000943110700086264501541916983708963470039740221935995651191587171109049754679615296537401913441904391993896149207689123204522227940630929598905457975559320524441869935715997774607730034609970592949807477206557618022329137052678703086363408827124715538988758491792396139587817134156958988552841965094845240579852518244271463497382541150608269789753147728935517533326248321816829180663120107925120203114411344625699291293763898514962433370407612481979139977092018927560720259091088697201556092650613019071224322963725833117316440820246666283047676776507209019763279291720643943889607533594707487239274327963487499538323662989431019046884226352937386343985306068691742310776363249614731488824345799922892983926012007305420971908334895957286359879083915868267033812582545634409470909263022188074830591940729115937626240993535742214427567244965017294865194615251000619376278571580393686714799014600811267987713640353107011690722853523116171702971354195670584288860851061635748686782785249925901522083281150832982397999950625384698560828067547435026163364315088952432560874657040210384541556389979932222856545440403668330781213360251455455120220822883737151062205912708055083302072994552466816276030892531396113059359287431376802927280311592226301322911802481409838447971844256537457868905786036320632962385014669351091094603318143806465691764752393176294641751679368375024112351030416911604300954976230362990064256904133648383450568748397056381269374194272359592392574692105490158217825831176438788100043610195583842687709608734168703348262722298499343036968942757119888162400068817121251947847879611411461403117049116361783698664730547841979758297296697145665086492905974146193271684797064701121935417990458660219000444964138474762216372319853824131663689957098670795263631647554808941313017500516057142745557424562423067770678754154499223065575787157328348480346417845563684053773582800480381779357601884782675988913470110837817214914593599868699992832912630690369931631549421385942886603658564727537405062453637962042550084569221287096924442810351378808288459432933267700821814757834415393902901285558326670847408421333400210124452461095597355496386036293473482290533361047367664209086306916936387007813280800785952850571307552795356292213476436682809833783141062958513207098732186394478303912749355429570513598957093741198833598573588120637333112663449648264732363623645146803860585286537971413557758553341665371055275555719211894896433181974883184183955988680858068602596896370734932527403518865096821484663599227404915315260901723092400451683840930500304800689426826583618560868616596777155559569938208220298418010930590098551783745776747977573742683245168022833147288922820965828286466821045092388373532567417752860749689852598778087643466401218885878144568817146803893879734608972074996874364401884666239882569104968294327287258003241545083810374862946680725991936416041994103639287227403748021131332390750377161464831342193337866306188260363718857207127151988065156154928645019346146449466729625423741399180189152393279815839737608554261824914490611307348009113053056006005689210543563539663341776397844035833208439729318844530891301513617954213705779506306067056662737410778537055262953723612663061943498377519208759264117424559447286324637463260964073673124017211489362682332981546647848242522774565632765828592990590982838949713282063043629840199407883531851047454873109105805647253759144823470992507355435259648438819662410682598873398645447287072404137279029036928170226815037865394102518179632451843695323029439398241746947488837483355359235207078911941067786776841606862801462768646054145087499448648319285999112535158401082766764639955681547573616389658378521527016238350227880073561528224562857951442289632816436998213690710485316946637138916834672167780786308942749672183242692282431729514116442821101872059120020199484366015957398970599787333324619855101436235197811657238689661581519653494003485369368186214004187140267005369780466181037109551014872587470017192777789485257799338423084258772921101939694966987189112170434940675366339462805302806815690375799822589857716366770160261713879643752541246474068453182313958449355875418337410424396936777732052384894780965431876415370725905887249226891779758692936743977175862862456472085396307898330805809155918914360029418930253999947844934069662128348548954459340109662151803467721057301827218110554034916427154901164664770476616074091988886653277341447720451784162704151334966271384964624753791855479385443870329979215196519014395325676116051615577312088441845609245638253418505774201653664329515921249207604453357340386154691620214828737961228580965415648344035861548772783156881243077199681785342248883847918525839705597679482388172405321942178758663195102131644994249872117966882373041751374818689211795222032414468208906906116183814132007140306039287827003093471808810672772636723972568623399771577345869074580372135660749245584199151643431876111229942487507824158629666584906444572556570723383723830896005572503089674716785782569141627752843716757420530936652370460422233631982675561319709848681306267230926106523122436075404318885410629058522561697130951229205628961666140956426283391464591201485872624572442140238902731974776879448009425833302793271348092644273789688513563743336880227833437076361285232661677073080974653028324899328918006703379268587550407982680995764368519859735216996762228879800124806101844381108925709527772330084786494263125014259698180318374634785431721717388172904505140133272488145053351081291660263689318106428567832060440848272965266798403207605991104876924750843645952584984440968613043569014246466386530217423786662023872793644091861275457436508959272613804635965393562749158935194157635474631195781660827843797221763434380089378409037124763917908785829196059077581504183926655839987644704137809126543493217498291707436163041018675585729107508932555063942659264687828673976669542067881158263526256622392398973273359993658648098146566346435609654428254820400937375561591708885231098713920362327080834342249338591358883703801938237897460912225647301407083719340974863730959229840631365849928565003006922720216392197008881549817895860780573794709634883164185971035095316968847885596030570259064889282484175883506394834562920590831895073625516405747114723914679497121447676403392608450414943502053056606823026631687243050849456328461187678547998299300155058610063786454411918703342359077917761878733462876047638778463913293409830775617061975760573805359269788969535737620788364427903297626712989344428189072411952658975243999722836280872604751987277307565451464340111942280401477341337061169345858248705703147554447073317482121170981035767923768876772449845257089957056409792334533231136840070247269083170223275666850973624005435502819612657028393978864763388747246594478138892875970720361109652986577619898410657587685297545136049831023385471520688746972243125727028102211859905057439882979247019899964631578661266807115642059840195682083463324212055692227058661640146924338260397847805429795099050856756788541356002753753399714708039727999100283647062528949390469429706255667807189563994612147321081024841359381986667344685526150371168837376118752825065328677899647212172817542299786308565035179577154527440605752256250641423122927122277959189606914335844995008286969362725712220292920513998221594208621538960534538907073579266666021103672121087483998879497419879362266948056738216012848493103932735419620327592319238702868527593052975885627457302321562291944488076164310010826736015043337164047438914990417331687674107644467966581015562267203370805908243171854175230601881539229980398059314383028440719449933783360603692745791113252012723201733115532373086683679775432942984850009109803700990470681703410561986076527208656101443242042480536928201148823421273313022157938788035914941282633723190214221697542428381554806013399185834891392579114562103995541911651089049093323933530933655041720520968248457783089580977878674939185174438438776671764967523170441726143335528587362577985767224290305302134798601943485448836733439735085560628664576365779443591301910910689409341947063922403424084167111172990341253306657889670466135051205017846537335530928867054618103441276265900216526355929382824306053701728906417476168280917019118271044045686275725165045244449737688303619106639945086304899767126994255593310009714962547115342947366078528651316326267515355027525276198767730507178457924853079888987721747182140490996366890200148490447971883428729507326259187714475227107439045772874972411296702430436328179737266281342409695073235465136943877075210318325533393261641158032214387911537911629590565711899313015401824525502661840556492343167546126590567058988090268643087438843139859126710384378375873807735592690608458468946394330560113029357605958764255960917145216751050973298935034795887710808405538011691936526778093520285457461275753271533858747021354010037800113740066881072184579523561036236734673622589308473592922188354921903136511777912203732726944524940966203103281674449051878500986260834575704914129824703509359659161901154653036497708320663554441497708002986744980186986264032359767267299496813545345485816192720063671886035020032412315758125950899440433435012432026012278686843470863760090777532730557584320672251809845096988015897417003431701038387578640152762884688329249465078277552995339862565306383165940542772798530947714367493828772772258925593906298608389248803197715578180078330969508817180139318122268922784842258911588698675641475978008552746720390573814830914810852780977605467454646379862668550277919235153590181943956752793061923311408120698770518612485548619080393645745534352829362179677519164207485271698440697272378163564403970588095956086711229283148582803450563628805516874960578766933824908077433240402560954920594206472903413779214999153626248221380593242277305313701007571268203374208283217484373241265418674521820549317546115057384729118430873249206203879903746683408952317250431573093025019075392010365039428204866167585262890354479220717088775109229173299608215836318633757488027569077974093239474555281902799772496731110211672542261744998109698766328635416702499365286083607867889094132028994027738720688247615972742867264595979006518864663014595276175184853178560869617919374066431952054554922203949547816245725880628914283488198207467797079341599411558618304256883435534524734574911817292573425656540944267842177533373106960609294533343385350359349969425846260530125248653222270759709364960310743059401339376800620736630703548468743030448616981004592061181482040182961373278313550627852892651218147420057259833138217651920769656119294617606558169196350801314687646365484290736635607282899756202569258967800268774037035709808156278740267693989813965678709031507247821667215671435438588267959067907609241590322780250274206268484656980963779086342466802350436024003212879855065086538531688777211493364901601311538006294071837668448630543810997821824271503874915410751241679606183174333143060982753117353482887601230806657598269534880642060264600253444416015340548479275900407730658937534242624298449833431462719063538237758061120620385352980854667275692596396570498922242736780324960637540374787886408715365473619277938728183291320504014772792990307753703165804352651864826517495604116398707340809463619266641303515161769118881607421324158173491831171317650943421359930632084276523279953886593715543602060355924365902579383711320838878256310369101227906946780223169718106653486657379292677359684727584313780501036988061400137595305453554615513580102853952206713460797147780307863881435083027559268993272478165836372015053049179918129455242540365041765403402628833752468776392974273313166208894631907695069501944379539094582515760766105389218833602057722362205001305870438266025308025417602404948855562182150750751106888403316928766483315584501348997280956797617745671575924866273336227481088468655407872159991066103600698390940480769763763226997871011844818306976538335005950336457065432457216539413080127678652862037434210688894830982162899392089268757360988526382270389455088342581444862648420627707771505180803187125787594184402376213089052126650061050469701281816042595737199734497806299103371494318828656848612378798811735110991799801057045960605780001709369448928769365028651996026578672678450079908163834929188519977754525714694596761486520432166997252390953466964124807162332712784884431444982591980518994630207287579758719790224808167006225626439367052162681433866331780512412344323014369067443821298706656540380265712213759871923873931362566435773738462560600364660305395294842893059070576924869505918462488468212064927732067583776130186731079289743628116599583455711448107308911906553131662338456789804734845344965686546984265271961224112133186629059890583108391524411051253100451040911956708456195347921341875782351232379993528423170189045198709469374010742150394295401234156455670830236325232079739653213193476064218443598317622307527745136709208838647203143672520934234631326419010789543506815122516411847853610396645922212933401857603333838976997211305421498742360190510327697296636267461722865241833251553598613026193751551319831380256044888332976831522248810071788263480784540623593403064844055316630145925301744630245157272475884017113107727478886074874015224307437959757053252571707331271245028097661489141667284440362806859858735454130104601326981512395713087286038808430237332886732839313885002707160846746242934522964057018781768931248275407123593356917776923533425997876112782099511285599353840778941314831502182630040212447108395967458602145035859781618655216527560192627708086087262387720822226889652730616685989072527695209635143257413389464945995662597035751967823525858032489608369793985312107770266060395590829142056643844055664005869322789378574881867310667605736329779641052882725274518827286625428382393459166554885963707401730024472278239706413440618742110947872355372838202600623493413113765961482147426437731511360786728665027139608889475767583243229961436891214074272597330447446786053965622390969587250636754547235050923782570214039780636525430368645923904096840115247880837541174360210096913287595285041359549556279023891109389097462281682322033943199312629407081464061362934339712088432385837505393037268193506672685874696321193636706337612104685155088742033277565581893722568507182915529057399597085076787642295525095975457903349150612304319565955197008899867486693823765271899019539568974321267273254882222224004755389239859979835100706365968743535527329690902859311344653516783599610560698671691184343054886885290729424763583545571423814381339783849620985549212837527039771087925224184327781363244495117451432573075873878093838134419343485390703531745241180852306550748341801915184859791021134978927558962902879173494067344369123401086764064315193604853990469544522908449409285948746871645840491219025588947776849057432541337374797794966053710927341200465715724763506857605312423344415248075754803033774274108373978240993812954391251333203582219231016318976597904238491187204688037978418856179982301879179764266276531172473040830748563170755336862974874506131984966755723619905268151938782423664299137505600721988442698369132110022015485352679774237782263853852463321813530639532439943791153262647456397525094667948124495460348890325171465539612499549963778925658115176018729026363852293177966848954873565000183306832761194647414213121210751632925932413707812549980362436995134299437015832978547461983675835105516924603112080088763435868456959335967723200779128207355339889925634470042038023616533569418329237199130375004191929039013427120726898257508552617542352441374938194991566182103972616620486339097078651247018721231827950542250106191746611009517207592111930479388912297056005114153250762412113260985752395338853242257777860647933927460186782789782699723973590265448805751667077594775072974923881049779012528749179830594593420922719837683070334641826581831923907252596631124852885273438860870784882162505742813611889002895484869496253948378515215131137719371620940957382640976520124562913526763760874386774704435039433409487060685204060888355379575110676610533649510112980637978422847069876550127179065142084028025796678539053891866124587294458964996246157456751960345161114351588819153537840962085022836026778571940383830796413794071892555933483322443090395701027676649566054884573649233426432962912619844388300723108540204508595112630732358101578179256409442819247667060055127334922228856321985898411582783988911273675381959390180023693683708490276060546797270803393665801040848846637263466786569710229153120519000963047314740374805265708657280792283723779745210946871811466182025660993311406611578983817640544811230301456784053996769926790335365543002023066218528705606392816137234640272921578618227747156717664165037746924371694071665288020381593326381182606418552631939009966082589498769111552075824111376123550105672956390868754379474006848086626986476298717298671815728591540315173245205436751668420180574145352660405905959587539963135743489979456271549976110549486567105694043094964989774575278389785530570332984739771251871262898585965517758103737494052230197359866090356416765677881934662507024128120700394949813325101634745599155156306247579069592768976528178629954872731049269714424393132267293825383287316287130607771650259666021181187766499046633855275695102350912480267554633885612378899021424128277998765426697370708042388286430046361608700541450549256559845993719814067504468551331402293233599428661964624075154913008661942644094902572103395590896050508204944087054475452645285336083685408288825888926541136287341761052361477259257422985904299442090737187450320617564810004346526971896066314134255790702920971076786902156598501598296937223739293268681830879457047395603956195577328890213672922040437812555939963419293936474314053621460130861297135564239379712465431552001373429780617437628498944042616477267774371765620436510287236482347810713681574878693736214510889065279486548240885089560262656290513652980665647483857787757493306066927416351802647706619250702512248087323968382228674852260351694304159466901141270386245309973893073452083256161180500489230969846777106979469537445664300966656124966292570906428861063638591045444116263389688245541779973114568448947820765373138369740288043085767453038721621136559037737238088971003570684171161501135376318445832359738593888308244869560883882833935382975557508302340927807157115083904806441387316403334988180194946369976699414405067155180621648698362241710049101372150396115632592492132900762935810877222601761760190598726498982134353646455008759410607362381609137770365298158662048183139267773602916034987537237880431390489674063610853859993162869263579709971544407313215896614504539798768874675974978868792287370772252188912876189961672993611728230210557385022798750835330949988432741337390786868012211128583699347625295665332548365419828774110469050554027152557171209808319362887843582492579714426009517567373319766313282873510002439034501656358247929921043889536551179573502304913607787703897702874485342232493848755060395543888343265692050269600087422277937808493437063360967690248327364296503714208133219129376307895265941549870117078901286890756332410992284924900738962889223659692135544094137272448620571101176051496165840665524384115587459309663538240550570486645065273370189346439666222040498168303073145567793660998897857009058989961503978190092877028722219004248978587810372943023569392986084229579715307464590674420891831787397395486007499924408066507131390352340947537887523127777175771196865526937196417712967419458824305622922437812948544960795268817934942798823408549254866263593125277677336363743369549244500071087631310949632393937947210294845177923018762571791974502871749456232719618788425967822000498560755499884589350732705244809671000916130387208779922584717888206871701180371702516512584769223934942452735654508085966577124113022033906578808120968629822409688682101059053873602450959246055223952410682383427245101109923844939734903406812732523210550188062850798255057157156415756773377519896354713455207287913285238287926335119227668015975487116022101446317196505076523859628621307606298034993460006270379886338045080658824326369835401659143996753784504909879448501812535866624953007980380806654056460696739471903127214281370321005791388835298981758596703260710729686277444723509614409795010882607453961605653275091062061574353073592710984531400419680014428565916101275718485582960777310722226010304546982606832089445628645393371511348506464504332053398402631192557349097148113431898393947473305047473337678551422611439646918176020144961741633856680037710981542356429020084648272350187808379741520836468785094514074137032018358799000084970078980665542357871577718113658221570928895649617895276312698239727552896235898582598699466590419749933139327883501632884022249783736923042565969465211191634530381791600541823233341675342632219344905536668626813028231444360711521765925701442530885666000518109528544182345032793183317574842675367549467294561088136429547749939817456462391505467318123273085199940080232803937337447914588770682640305956395814447182408634493188521453495558055198034696424443890979502437966017245640940998472551715991165414959406837064688451094354451690140357198798458386719603743675043330588582295891995873015987801882769555153095268001351516074940423576793694041191360763477925597374800262890519827115116412758077940649183631001280353712823992231967865144614626166752637798063970911960990884013849134904498849748041073280313182960206519042881905320728243506381028139925653715929056225651035081656193868958290597853283887362584488170880133923977716265930501920677099037293955438589944960404689978988796467148157310120570435638388395393500306300897404065935462840229194866306017333148638533604802610828792729292944394773047137328223634789149573576063531060499525820476912028518268280907761034264568816039199074277408109684873137389641982176399077837359909977214747530960788685225039518003056603977693122368173655438551206066294869096770628705334708895176851919611452056279786373081274498329607449730207225156113979530231294630970365246945501116272946293659318012809299917418018766324894707468104124220611626259651859851115732104521313286496576167137264848036963832137421385191353589682364449627364910725518112197263961716268922209530294251485762370632610933065411636744228267424204615489099665211539283053041319521995819316838124769981544337348684692502606268787148021849570816008091416823832956776722849374148302528191771056521665362958444295522994652646655501312784515773923042049584204027042279355959305395015371650979216598442409282674555355471241886431247235403335618574831035977231962919629617652954689765993253523608777004780296999535509762728082818910673674968496678542172899191114411247395817209105243360370053626904531207600238042495645263902356282700318794919182645739711305977296693674431289690564252208634652104390636246683978710364311944136894210086185442657035098470078355564999686582459943980945289418862057642947978160249778972409269201489337573523151832846086725892385574561247629333134647452140686245034528533261410418924119455874648994411937947936407244743384631338659180887482397427163791573963887165901055696278263843
;10 13104035107217212816165471425010639730009541565951152867433739818916352/6610142950363663847570004456545448869985776325831404063642721608186439846750169
